<?php
/**
 * Plugin Name: ShortURL Integration
 * Plugin URI: https://yoursite.com/shorturl
 * Description: Integrate ShortURL link shortening service into WordPress. Create short URLs directly from your dashboard.
 * Version: 1.0.0
 * Author: ShortURL
 * Author URI: https://yoursite.com
 * License: GPL v2 or later
 * Text Domain: shorturl-integration
 */

if (!defined('ABSPATH')) {
    exit;
}

define('SHORTURL_PLUGIN_VERSION', '1.0.0');
define('SHORTURL_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('SHORTURL_PLUGIN_URL', plugin_dir_url(__FILE__));

// Activation/Deactivation hooks
register_activation_hook(__FILE__, 'shorturl_plugin_activate');
register_deactivation_hook(__FILE__, 'shorturl_plugin_deactivate');

function shorturl_plugin_activate() {
    $api_url = get_option('shorturl_api_url', '');
    if ($api_url) {
        $track_url = rtrim($api_url, '/') . '/api/plugin-track.php?action=activate';
        wp_remote_post($track_url, [
            'timeout' => 5,
            'blocking' => false,
            'body' => json_encode([
                'site_url' => home_url(),
                'site_name' => get_bloginfo('name'),
                'wp_version' => get_bloginfo('version'),
                'plugin_version' => SHORTURL_PLUGIN_VERSION,
                'php_version' => phpversion()
            ]),
            'headers' => ['Content-Type' => 'application/json']
        ]);
    }
}

function shorturl_plugin_deactivate() {
    $api_url = get_option('shorturl_api_url', '');
    if ($api_url) {
        $track_url = rtrim($api_url, '/') . '/api/plugin-track.php?action=deactivate';
        wp_remote_post($track_url, [
            'timeout' => 5,
            'blocking' => false,
            'body' => json_encode(['site_url' => home_url()]),
            'headers' => ['Content-Type' => 'application/json']
        ]);
    }
}

class ShortURL_Integration {
    
    private static $instance = null;
    private $api_url;
    private $api_key;
    
    public static function instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        $this->api_url = get_option('shorturl_api_url', '');
        $this->api_key = get_option('shorturl_api_key', '');
        
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_scripts']);
        
        // Weekly ping to track active installations
        add_action('shorturl_weekly_ping', [$this, 'send_tracking_ping']);
        if (!wp_next_scheduled('shorturl_weekly_ping')) {
            wp_schedule_event(time(), 'weekly', 'shorturl_weekly_ping');
        }
        
        // Shortcode
        add_shortcode('shorturl', [$this, 'shortcode_handler']);
        
        // Gutenberg block
        add_action('init', [$this, 'register_block']);
        
        // Meta box for posts/pages
        add_action('add_meta_boxes', [$this, 'add_meta_box']);
        add_action('save_post', [$this, 'save_meta_box']);
        
        // AJAX handlers
        add_action('wp_ajax_shorturl_create', [$this, 'ajax_create_short_url']);
        add_action('wp_ajax_shorturl_get_stats', [$this, 'ajax_get_stats']);
        
        // Widget
        add_action('widgets_init', [$this, 'register_widget']);
    }
    
    public function add_admin_menu() {
        add_menu_page(
            'ShortURL',
            'ShortURL',
            'manage_options',
            'shorturl',
            [$this, 'render_dashboard_page'],
            'dashicons-admin-links',
            30
        );
        
        add_submenu_page(
            'shorturl',
            'Dashboard',
            'Dashboard',
            'manage_options',
            'shorturl',
            [$this, 'render_dashboard_page']
        );
        
        add_submenu_page(
            'shorturl',
            'Create Link',
            'Create Link',
            'edit_posts',
            'shorturl-create',
            [$this, 'render_create_page']
        );
        
        add_submenu_page(
            'shorturl',
            'Settings',
            'Settings',
            'manage_options',
            'shorturl-settings',
            [$this, 'render_settings_page']
        );
    }
    
    public function register_settings() {
        register_setting('shorturl_settings', 'shorturl_api_url');
        register_setting('shorturl_settings', 'shorturl_api_key');
        register_setting('shorturl_settings', 'shorturl_auto_shorten');
        
        // Send tracking ping when settings are saved
        if (isset($_POST['shorturl_api_url']) && !empty($_POST['shorturl_api_url'])) {
            $this->send_tracking_ping();
        }
    }
    
    public function send_tracking_ping() {
        if (!$this->api_url) {
            return;
        }
        
        $track_url = rtrim($this->api_url, '/');
        // Handle both /api/v1 and base URL formats
        $track_url = preg_replace('#/api/v1/?$#', '', $track_url);
        $track_url .= '/api/plugin-track.php?action=ping';
        
        wp_remote_post($track_url, [
            'timeout' => 5,
            'blocking' => false,
            'body' => json_encode([
                'site_url' => home_url(),
                'site_name' => get_bloginfo('name'),
                'wp_version' => get_bloginfo('version'),
                'plugin_version' => SHORTURL_PLUGIN_VERSION,
                'php_version' => phpversion()
            ]),
            'headers' => ['Content-Type' => 'application/json']
        ]);
    }
    
    public function enqueue_admin_scripts($hook) {
        if (strpos($hook, 'shorturl') === false) {
            return;
        }
        
        wp_enqueue_style('shorturl-admin', SHORTURL_PLUGIN_URL . 'assets/admin.css', [], SHORTURL_PLUGIN_VERSION);
        wp_enqueue_script('shorturl-admin', SHORTURL_PLUGIN_URL . 'assets/admin.js', ['jquery'], SHORTURL_PLUGIN_VERSION, true);
        
        wp_localize_script('shorturl-admin', 'shorturl_ajax', [
            'url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('shorturl_nonce')
        ]);
    }
    
    public function render_settings_page() {
        ?>
        <div class="wrap">
            <h1>ShortURL Settings</h1>
            
            <form method="post" action="options.php">
                <?php settings_fields('shorturl_settings'); ?>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">API URL</th>
                        <td>
                            <input type="url" name="shorturl_api_url" value="<?php echo esc_attr(get_option('shorturl_api_url')); ?>" class="regular-text" placeholder="https://yoursite.com/shorturl/api/v1">
                            <p class="description">Your ShortURL installation API endpoint</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">API Key</th>
                        <td>
                            <input type="text" name="shorturl_api_key" value="<?php echo esc_attr(get_option('shorturl_api_key')); ?>" class="regular-text">
                            <p class="description">Your API key from ShortURL account settings</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Auto-shorten Links</th>
                        <td>
                            <label>
                                <input type="checkbox" name="shorturl_auto_shorten" value="1" <?php checked(get_option('shorturl_auto_shorten'), 1); ?>>
                                Automatically shorten external links in posts
                            </label>
                        </td>
                    </tr>
                </table>
                
                <?php submit_button(); ?>
            </form>
            
            <?php if ($this->api_url && $this->api_key): ?>
            <h2>Connection Test</h2>
            <button type="button" id="shorturl-test-connection" class="button">Test Connection</button>
            <span id="shorturl-test-result"></span>
            <?php endif; ?>
        </div>
        <?php
    }
    
    public function render_dashboard_page() {
        if (!$this->api_url || !$this->api_key) {
            echo '<div class="wrap"><h1>ShortURL Dashboard</h1>';
            echo '<div class="notice notice-warning"><p>Please configure your API settings first. <a href="' . admin_url('admin.php?page=shorturl-settings') . '">Go to Settings</a></p></div>';
            echo '</div>';
            return;
        }
        
        $stats = $this->api_request('account');
        ?>
        <div class="wrap">
            <h1>ShortURL Dashboard</h1>
            
            <?php if (is_wp_error($stats)): ?>
            <div class="notice notice-error"><p>Error connecting to API: <?php echo esc_html($stats->get_error_message()); ?></p></div>
            <?php else: ?>
            
            <div class="shorturl-stats-grid">
                <div class="shorturl-stat-box">
                    <h3>Total Links</h3>
                    <span class="stat-number"><?php echo number_format($stats['data']['usage']['total_links'] ?? 0); ?></span>
                </div>
                <div class="shorturl-stat-box">
                    <h3>Total Clicks</h3>
                    <span class="stat-number"><?php echo number_format($stats['data']['usage']['total_clicks'] ?? 0); ?></span>
                </div>
                <div class="shorturl-stat-box">
                    <h3>This Month</h3>
                    <span class="stat-number"><?php echo number_format($stats['data']['usage']['links_this_month'] ?? 0); ?></span>
                </div>
                <div class="shorturl-stat-box">
                    <h3>Plan</h3>
                    <span class="stat-number"><?php echo esc_html($stats['data']['plan']['name'] ?? 'Free'); ?></span>
                </div>
            </div>
            
            <h2>Quick Create</h2>
            <form id="shorturl-quick-create" class="shorturl-create-form">
                <input type="url" name="url" placeholder="Enter URL to shorten..." required>
                <input type="text" name="title" placeholder="Title (optional)">
                <button type="submit" class="button button-primary">Shorten</button>
            </form>
            <div id="shorturl-result"></div>
            
            <?php endif; ?>
        </div>
        <?php
    }
    
    public function render_create_page() {
        ?>
        <div class="wrap">
            <h1>Create Short Link</h1>
            
            <form id="shorturl-create-form" class="shorturl-create-form-full">
                <table class="form-table">
                    <tr>
                        <th>Destination URL *</th>
                        <td><input type="url" name="url" class="large-text" required></td>
                    </tr>
                    <tr>
                        <th>Title</th>
                        <td><input type="text" name="title" class="regular-text"></td>
                    </tr>
                    <tr>
                        <th>Custom Alias</th>
                        <td><input type="text" name="custom_alias" class="regular-text" placeholder="my-custom-link"></td>
                    </tr>
                </table>
                
                <p class="submit">
                    <button type="submit" class="button button-primary button-large">Create Short Link</button>
                </p>
            </form>
            
            <div id="shorturl-create-result"></div>
        </div>
        <?php
    }
    
    public function shortcode_handler($atts) {
        $atts = shortcode_atts([
            'url' => '',
            'text' => '',
            'class' => ''
        ], $atts);
        
        if (empty($atts['url'])) {
            return '';
        }
        
        // Check cache first
        $cache_key = 'shorturl_' . md5($atts['url']);
        $short_url = get_transient($cache_key);
        
        if (!$short_url) {
            $result = $this->create_short_url($atts['url']);
            if (!is_wp_error($result)) {
                $short_url = $result['short_url'];
                set_transient($cache_key, $short_url, DAY_IN_SECONDS);
            } else {
                return '<a href="' . esc_url($atts['url']) . '">' . esc_html($atts['text'] ?: $atts['url']) . '</a>';
            }
        }
        
        $text = $atts['text'] ?: $short_url;
        $class = $atts['class'] ? ' class="' . esc_attr($atts['class']) . '"' : '';
        
        return '<a href="' . esc_url($short_url) . '"' . $class . '>' . esc_html($text) . '</a>';
    }
    
    public function register_block() {
        if (!function_exists('register_block_type')) {
            return;
        }
        
        wp_register_script(
            'shorturl-block',
            SHORTURL_PLUGIN_URL . 'assets/block.js',
            ['wp-blocks', 'wp-element', 'wp-editor', 'wp-components'],
            SHORTURL_PLUGIN_VERSION
        );
        
        register_block_type('shorturl/link', [
            'editor_script' => 'shorturl-block',
            'render_callback' => [$this, 'render_block']
        ]);
    }
    
    public function render_block($attributes) {
        $url = $attributes['url'] ?? '';
        $text = $attributes['text'] ?? '';
        
        return $this->shortcode_handler(['url' => $url, 'text' => $text]);
    }
    
    public function add_meta_box() {
        add_meta_box(
            'shorturl_meta_box',
            'ShortURL',
            [$this, 'render_meta_box'],
            ['post', 'page'],
            'side',
            'default'
        );
    }
    
    public function render_meta_box($post) {
        $short_url = get_post_meta($post->ID, '_shorturl_link', true);
        $permalink = get_permalink($post->ID);
        ?>
        <div class="shorturl-meta-box">
            <?php if ($short_url): ?>
            <p><strong>Short URL:</strong></p>
            <input type="text" value="<?php echo esc_url($short_url); ?>" readonly class="widefat" onclick="this.select();">
            <p><button type="button" class="button shorturl-copy-btn" data-url="<?php echo esc_url($short_url); ?>">Copy</button></p>
            <?php else: ?>
            <p>Create a short URL for this <?php echo $post->post_type; ?>:</p>
            <button type="button" class="button button-primary shorturl-create-for-post" data-url="<?php echo esc_url($permalink); ?>" data-post-id="<?php echo $post->ID; ?>">
                Generate Short URL
            </button>
            <?php endif; ?>
        </div>
        <?php
        wp_nonce_field('shorturl_meta_box', 'shorturl_meta_nonce');
    }
    
    public function save_meta_box($post_id) {
        if (!isset($_POST['shorturl_meta_nonce']) || !wp_verify_nonce($_POST['shorturl_meta_nonce'], 'shorturl_meta_box')) {
            return;
        }
        
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
    }
    
    public function ajax_create_short_url() {
        check_ajax_referer('shorturl_nonce', 'nonce');
        
        $url = sanitize_url($_POST['url'] ?? '');
        $title = sanitize_text_field($_POST['title'] ?? '');
        $custom_alias = sanitize_text_field($_POST['custom_alias'] ?? '');
        $post_id = intval($_POST['post_id'] ?? 0);
        
        if (empty($url)) {
            wp_send_json_error(['message' => 'URL is required']);
        }
        
        $data = ['url' => $url];
        if ($title) $data['title'] = $title;
        if ($custom_alias) $data['custom_alias'] = $custom_alias;
        
        $result = $this->api_request('links', 'POST', $data);
        
        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        }
        
        // Save to post meta if post_id provided
        if ($post_id && isset($result['data']['short_url'])) {
            update_post_meta($post_id, '_shorturl_link', $result['data']['short_url']);
        }
        
        wp_send_json_success($result['data']);
    }
    
    public function ajax_get_stats() {
        check_ajax_referer('shorturl_nonce', 'nonce');
        
        $result = $this->api_request('account');
        
        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        }
        
        wp_send_json_success($result['data']);
    }
    
    public function register_widget() {
        register_widget('ShortURL_Widget');
    }
    
    private function create_short_url($url, $options = []) {
        $data = array_merge(['url' => $url], $options);
        $result = $this->api_request('links', 'POST', $data);
        
        if (is_wp_error($result)) {
            return $result;
        }
        
        return $result['data'] ?? null;
    }
    
    private function api_request($endpoint, $method = 'GET', $data = null) {
        if (!$this->api_url || !$this->api_key) {
            return new WP_Error('not_configured', 'API not configured');
        }
        
        $url = trailingslashit($this->api_url) . $endpoint;
        
        $args = [
            'method' => $method,
            'headers' => [
                'X-API-Key' => $this->api_key,
                'Content-Type' => 'application/json'
            ],
            'timeout' => 30
        ];
        
        if ($data && in_array($method, ['POST', 'PUT'])) {
            $args['body'] = json_encode($data);
        }
        
        $response = wp_remote_request($url, $args);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $body = wp_remote_retrieve_body($response);
        $result = json_decode($body, true);
        
        if (wp_remote_retrieve_response_code($response) >= 400) {
            return new WP_Error('api_error', $result['error'] ?? 'API request failed');
        }
        
        return $result;
    }
}

// Widget Class
class ShortURL_Widget extends WP_Widget {
    
    public function __construct() {
        parent::__construct(
            'shorturl_widget',
            'ShortURL Quick Shorten',
            ['description' => 'Allow visitors to shorten URLs']
        );
    }
    
    public function widget($args, $instance) {
        echo $args['before_widget'];
        
        if (!empty($instance['title'])) {
            echo $args['before_title'] . apply_filters('widget_title', $instance['title']) . $args['after_title'];
        }
        
        ?>
        <form class="shorturl-widget-form" method="post">
            <input type="url" name="url" placeholder="Enter URL..." required>
            <button type="submit">Shorten</button>
        </form>
        <div class="shorturl-widget-result"></div>
        <?php
        
        echo $args['after_widget'];
    }
    
    public function form($instance) {
        $title = !empty($instance['title']) ? $instance['title'] : 'Shorten URL';
        ?>
        <p>
            <label for="<?php echo esc_attr($this->get_field_id('title')); ?>">Title:</label>
            <input class="widefat" id="<?php echo esc_attr($this->get_field_id('title')); ?>" name="<?php echo esc_attr($this->get_field_name('title')); ?>" type="text" value="<?php echo esc_attr($title); ?>">
        </p>
        <?php
    }
    
    public function update($new_instance, $old_instance) {
        $instance = [];
        $instance['title'] = (!empty($new_instance['title'])) ? sanitize_text_field($new_instance['title']) : '';
        return $instance;
    }
}

// Initialize plugin
ShortURL_Integration::instance();
