/**
 * ShortURL Browser Extension - Popup Script
 */

// Configuration
const DEFAULT_API_URL = 'http://localhost/shorturl/api/v1';

// State
let apiUrl = DEFAULT_API_URL;
let apiKey = null;

// DOM Elements
const loading = document.getElementById('loading');
const loggedOut = document.getElementById('loggedOut');
const loggedIn = document.getElementById('loggedIn');
const shortener = document.getElementById('shortener');
const urlInput = document.getElementById('urlInput');
const aliasInput = document.getElementById('aliasInput');
const shortenBtn = document.getElementById('shortenBtn');
const result = document.getElementById('result');
const shortUrl = document.getElementById('shortUrl');
const copyBtn = document.getElementById('copyBtn');
const openBtn = document.getElementById('openBtn');
const error = document.getElementById('error');
const totalLinks = document.getElementById('totalLinks');
const totalClicks = document.getElementById('totalClicks');
const loginBtn = document.getElementById('loginBtn');
const dashboardLink = document.getElementById('dashboardLink');
const optionsLink = document.getElementById('optionsLink');

// Initialize
document.addEventListener('DOMContentLoaded', init);

async function init() {
  // Load settings
  const settings = await chrome.storage.sync.get(['apiUrl', 'apiKey']);
  apiUrl = settings.apiUrl || DEFAULT_API_URL;
  apiKey = settings.apiKey || null;
  
  // Update dashboard link
  const baseUrl = apiUrl.replace('/api/v1', '');
  dashboardLink.href = baseUrl + '/dashboard.php';
  loginBtn.onclick = () => chrome.tabs.create({ url: baseUrl + '/login.php' });
  optionsLink.onclick = (e) => {
    e.preventDefault();
    chrome.runtime.openOptionsPage();
  };
  
  // Check authentication
  if (apiKey) {
    await checkAuth();
  } else {
    showLoggedOut();
  }
  
  // Get current tab URL
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  if (tab && tab.url && !tab.url.startsWith('chrome://')) {
    urlInput.value = tab.url;
  }
  
  // Event listeners
  shortenBtn.addEventListener('click', shortenUrl);
  copyBtn.addEventListener('click', copyToClipboard);
  openBtn.addEventListener('click', openShortUrl);
  
  urlInput.addEventListener('keypress', (e) => {
    if (e.key === 'Enter') shortenUrl();
  });
}

async function checkAuth() {
  try {
    const response = await fetch(`${apiUrl}/account`, {
      headers: { 'X-API-Key': apiKey }
    });
    
    if (response.ok) {
      const data = await response.json();
      showLoggedIn();
      loadStats();
    } else {
      showLoggedOut();
    }
  } catch (err) {
    showLoggedOut();
  }
}

function showLoading() {
  loading.style.display = 'block';
  loggedOut.style.display = 'none';
  loggedIn.style.display = 'none';
}

function showLoggedOut() {
  loading.style.display = 'none';
  loggedOut.style.display = 'block';
  loggedIn.style.display = 'none';
}

function showLoggedIn() {
  loading.style.display = 'none';
  loggedOut.style.display = 'none';
  loggedIn.style.display = 'block';
  shortener.style.display = 'block';
}

async function loadStats() {
  try {
    const response = await fetch(`${apiUrl}/analytics/account`, {
      headers: { 'X-API-Key': apiKey }
    });
    
    if (response.ok) {
      const data = await response.json();
      totalLinks.textContent = formatNumber(data.total_links || 0);
      totalClicks.textContent = formatNumber(data.total_clicks || 0);
    }
  } catch (err) {
    console.error('Failed to load stats:', err);
  }
}

async function shortenUrl() {
  const url = urlInput.value.trim();
  const alias = aliasInput.value.trim();
  
  if (!url) {
    showError('Please enter a URL');
    return;
  }
  
  if (!isValidUrl(url)) {
    showError('Please enter a valid URL');
    return;
  }
  
  shortenBtn.disabled = true;
  shortenBtn.textContent = 'Shortening...';
  hideError();
  result.style.display = 'none';
  
  try {
    const body = { url };
    if (alias) body.custom_alias = alias;
    
    const response = await fetch(`${apiUrl}/links`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-API-Key': apiKey
      },
      body: JSON.stringify(body)
    });
    
    const data = await response.json();
    
    if (response.ok && data.short_url) {
      shortUrl.textContent = data.short_url;
      result.style.display = 'block';
      loadStats(); // Refresh stats
    } else {
      showError(data.error || 'Failed to shorten URL');
    }
  } catch (err) {
    showError('Network error. Please try again.');
  } finally {
    shortenBtn.disabled = false;
    shortenBtn.textContent = 'Shorten URL';
  }
}

function copyToClipboard() {
  const url = shortUrl.textContent;
  navigator.clipboard.writeText(url).then(() => {
    copyBtn.textContent = 'Copied!';
    setTimeout(() => {
      copyBtn.textContent = 'Copy';
    }, 2000);
  });
}

function openShortUrl() {
  const url = shortUrl.textContent;
  chrome.tabs.create({ url });
}

function showError(message) {
  error.textContent = message;
  error.style.display = 'block';
}

function hideError() {
  error.style.display = 'none';
}

function isValidUrl(string) {
  try {
    new URL(string);
    return true;
  } catch (_) {
    return false;
  }
}

function formatNumber(num) {
  if (num >= 1000000) {
    return (num / 1000000).toFixed(1) + 'M';
  }
  if (num >= 1000) {
    return (num / 1000).toFixed(1) + 'K';
  }
  return num.toString();
}
