/**
 * ShortURL Browser Extension - Background Service Worker
 */

// Context menu for right-click shortening
chrome.runtime.onInstalled.addListener(() => {
  chrome.contextMenus.create({
    id: 'shortenLink',
    title: 'Shorten this link with ShortURL',
    contexts: ['link']
  });
  
  chrome.contextMenus.create({
    id: 'shortenPage',
    title: 'Shorten current page with ShortURL',
    contexts: ['page']
  });
});

// Handle context menu clicks
chrome.contextMenus.onClicked.addListener(async (info, tab) => {
  const settings = await chrome.storage.sync.get(['apiUrl', 'apiKey']);
  const apiUrl = settings.apiUrl || 'http://localhost/shorturl/api/v1';
  const apiKey = settings.apiKey;
  
  if (!apiKey) {
    chrome.notifications.create({
      type: 'basic',
      iconUrl: 'icons/icon48.png',
      title: 'ShortURL',
      message: 'Please sign in to use ShortURL'
    });
    return;
  }
  
  let urlToShorten;
  
  if (info.menuItemId === 'shortenLink') {
    urlToShorten = info.linkUrl;
  } else if (info.menuItemId === 'shortenPage') {
    urlToShorten = info.pageUrl || tab.url;
  }
  
  if (urlToShorten) {
    try {
      const response = await fetch(`${apiUrl}/links`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-API-Key': apiKey
        },
        body: JSON.stringify({ url: urlToShorten })
      });
      
      const data = await response.json();
      
      if (response.ok && data.short_url) {
        // Copy to clipboard
        await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          func: (text) => navigator.clipboard.writeText(text),
          args: [data.short_url]
        });
        
        chrome.notifications.create({
          type: 'basic',
          iconUrl: 'icons/icon48.png',
          title: 'Link Shortened!',
          message: `${data.short_url} copied to clipboard`
        });
      } else {
        chrome.notifications.create({
          type: 'basic',
          iconUrl: 'icons/icon48.png',
          title: 'Error',
          message: data.error || 'Failed to shorten URL'
        });
      }
    } catch (err) {
      chrome.notifications.create({
        type: 'basic',
        iconUrl: 'icons/icon48.png',
        title: 'Error',
        message: 'Network error. Please try again.'
      });
    }
  }
});

// Listen for messages from popup
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'getSettings') {
    chrome.storage.sync.get(['apiUrl', 'apiKey'], sendResponse);
    return true;
  }
  
  if (request.action === 'saveSettings') {
    chrome.storage.sync.set(request.settings, () => {
      sendResponse({ success: true });
    });
    return true;
  }
});
